// routes/reviews.ts
import { Hono } from "hono";
import { zValidator } from "@hono/zod-validator";
import { requireAuth } from "@/lib/auth.js";
import { z } from "zod";
import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const reviewsRouter = new Hono<ContextVars>();

reviewsRouter.use("*", requireAuth, withPrisma);

// Schemas
const createReviewSchema = z.object({
  productId: z.number(),
  rating: z.number().min(1).max(5),
  title: z.string().optional(),
  comment: z.string().optional(),
});

const updateReviewSchema = createReviewSchema.partial();

/* ------------------------- CREATE REVIEW ------------------------- */
reviewsRouter.post("/", zValidator("json", createReviewSchema), async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const { productId, rating, title, comment } = c.req.valid("json");

  try {
    // Check if user has purchased the product
    const hasPurchased = await prisma.orderItem.findFirst({
      where: {
        productId,
        order: {
          userId: user.id,
          status: "DELIVERED",
        },
      },
    });

    if (!hasPurchased) {
      return c.json(
        { error: "You can only review products you have purchased" },
        400
      );
    }

    // Check if user already reviewed this product
    const existingReview = await prisma.review.findUnique({
      where: {
        userId_productId: {
          userId: user.id,
          productId,
        },
      },
    });

    if (existingReview) {
      return c.json({ error: "You have already reviewed this product" }, 400);
    }

    const review = await prisma.review.create({
      data: {
        userId: user.id,
        productId,
        rating,
        title,
        comment,
        isVerified: true, // Since we verified purchase
      },
      include: {
        user: {
          select: { name: true },
        },
        product: {
          select: { name: true, slug: true },
        },
      },
    });

    return c.json({ review }, 201);
  } catch (error) {
    console.error("Create review error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET USER REVIEWS ------------------------- */
reviewsRouter.get("/my-reviews", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const page = parseInt(c.req.query("page") || "1");
  const limit = parseInt(c.req.query("limit") || "10");

  try {
    const skip = (page - 1) * limit;

    const [reviews, total] = await Promise.all([
      prisma.review.findMany({
        where: { userId: user.id },
        skip,
        take: limit,
        include: {
          product: {
            include: {
              images: {
                where: { isPrimary: true },
                take: 1,
              },
            },
          },
        },
        orderBy: { createdAt: "desc" },
      }),
      prisma.review.count({
        where: { userId: user.id },
      }),
    ]);

    return c.json({
      reviews,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error("Get user reviews error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- UPDATE REVIEW ------------------------- */
reviewsRouter.put("/:id", zValidator("json", updateReviewSchema), async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const reviewId = parseInt(c.req.param("id"));
  const data = c.req.valid("json");

  if (isNaN(reviewId)) {
    return c.json({ error: "Invalid review ID" }, 400);
  }

  try {
    // Verify user owns the review
    const review = await prisma.review.findFirst({
      where: {
        id: reviewId,
        userId: user.id,
      },
    });

    if (!review) {
      return c.json({ error: "Review not found" }, 404);
    }

    const updatedReview = await prisma.review.update({
      where: { id: reviewId },
      data,
      include: {
        user: {
          select: { name: true },
        },
        product: {
          select: { name: true, slug: true },
        },
      },
    });

    return c.json({ review: updatedReview });
  } catch (error) {
    console.error("Update review error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- DELETE REVIEW ------------------------- */
reviewsRouter.delete("/:id", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const reviewId = parseInt(c.req.param("id"));

  if (isNaN(reviewId)) {
    return c.json({ error: "Invalid review ID" }, 400);
  }

  try {
    // Verify user owns the review
    const review = await prisma.review.findFirst({
      where: {
        id: reviewId,
        userId: user.id,
      },
    });

    if (!review) {
      return c.json({ error: "Review not found" }, 404);
    }

    await prisma.review.delete({
      where: { id: reviewId },
    });

    return c.json({ message: "Review deleted successfully" });
  } catch (error) {
    console.error("Delete review error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default reviewsRouter;
