// routes/public/products.ts
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const publicProductsRouter = new Hono<ContextVars>();

// Schema for query validation
const productQuerySchema = z.object({
  page: z.string().transform(Number).optional().default(1),
  limit: z.string().transform(Number).optional().default(20),
  category: z.string().optional(),
  search: z.string().optional(),
  minPrice: z.string().transform(Number).optional(),
  maxPrice: z.string().transform(Number).optional(),
  sortBy: z
    .enum(["name", "price", "createdAt", "popularity"])
    .optional()
    .default("createdAt"),
  sortOrder: z.enum(["asc", "desc"]).optional().default("desc"),
});

publicProductsRouter.use("*", withPrisma);
/* ------------------------- LIST PRODUCTS ------------------------- */
publicProductsRouter.get(
  "/",

  zValidator("query", productQuerySchema),
  async (c) => {
    try {
      const {
        page,
        limit,
        category,
        search,
        minPrice,
        maxPrice,
        sortBy,
        sortOrder,
      } = c.req.valid("query");

      const prisma = c.get("prisma");

      const skip = (page - 1) * limit;

      // Build where clause
      const where: any = {
        status: "ACTIVE", // Only show active products
      };

      if (category) {
        where.category = {
          OR: [
            { slug: category },
            { id: isNaN(Number(category)) ? undefined : Number(category) },
          ].filter(Boolean),
        };
      }

      if (search) {
        where.OR = [
          { name: { contains: search, mode: "insensitive" } },
          { description: { contains: search, mode: "insensitive" } },
          {
            tags: {
              some: {
                tag: { name: { contains: search, mode: "insensitive" } },
              },
            },
          },
        ];
      }

      if (minPrice !== undefined || maxPrice !== undefined) {
        where.AND = [];
        if (minPrice !== undefined)
          where.AND.push({ price: { gte: minPrice } });
        if (maxPrice !== undefined)
          where.AND.push({ price: { lte: maxPrice } });
      }

      // Build orderBy
      const orderBy: any = {};
      if (sortBy === "popularity") {
        // You might want to implement popularity based on sales/reviews
        orderBy.createdAt = sortOrder;
      } else {
        orderBy[sortBy] = sortOrder;
      }

      const [products, total] = await Promise.all([
        prisma.product.findMany({
          where,
          skip,
          take: limit,
          orderBy,
          include: {
            category: {
              select: { id: true, name: true, slug: true },
            },
            images: {
              where: { isPrimary: true },
              take: 1,
            },
            variants: {
              where: { stock: { gt: 0 } },
              select: { id: true, name: true, price: true, stock: true },
            },
            reviews: {
              select: { rating: true },
            },
            _count: {
              select: { reviews: true },
            },
          },
        }),
        prisma.product.count({ where }),
      ]);

      // Calculate average rating for each product
      const productsWithRating = products.map((product) => {
        const avgRating =
          product.reviews.length > 0
            ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
              product.reviews.length
            : 0;

        const { reviews, ...productWithoutReviews } = product;
        return {
          ...productWithoutReviews,
          averageRating: Math.round(avgRating * 10) / 10,
          reviewCount: product._count.reviews,
        };
      });

      return c.json({
        products: productsWithRating,
        pagination: {
          page,
          limit,
          total,
          pages: Math.ceil(total / limit),
          hasNext: page * limit < total,
          hasPrev: page > 1,
        },
      });
    } catch (error) {
      console.error("Get products error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- GET PRODUCT BY SLUG ------------------------- */
publicProductsRouter.get("/slug/:slug", async (c) => {
  try {
    const slug = c.req.param("slug");
    const prisma = c.get("prisma");

    const product = await prisma.product.findUnique({
      where: {
        slug,
        status: "ACTIVE",
      },
      include: {
        category: {
          select: { id: true, name: true, slug: true },
        },
        images: {
          orderBy: { isPrimary: "desc" },
        },
        variants: {
          where: { stock: { gt: 0 } },
          orderBy: { price: "asc" },
        },
        attributes: {
          include: {
            attribute: true,
            attributeValue: true,
          },
        },
        tags: {
          include: {
            tag: true,
          },
        },
        reviews: {
          where: { isVerified: true },
          include: {
            user: {
              select: { name: true },
            },
          },
          orderBy: { createdAt: "desc" },
        },
        _count: {
          select: { reviews: true },
        },
      },
    });

    if (!product) {
      return c.json({ error: "Product not found" }, 404);
    }

    // Calculate average rating
    const avgRating =
      product.reviews.length > 0
        ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
          product.reviews.length
        : 0;

    const { reviews, ...productWithoutReviews } = product;

    return c.json({
      ...productWithoutReviews,
      averageRating: Math.round(avgRating * 10) / 10,
      reviews: reviews.slice(0, 10), // Limit reviews in initial response
    });
  } catch (error) {
    console.error("Get product by slug error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET PRODUCT BY ID ------------------------- */
publicProductsRouter.get("/:id", async (c) => {
  try {
    const id = parseInt(c.req.param("id"));
    const prisma = c.get("prisma");

    if (isNaN(id)) {
      return c.json({ error: "Invalid product ID" }, 400);
    }

    const product = await prisma.product.findUnique({
      where: {
        id,
        status: "ACTIVE",
      },
      include: {
        category: {
          select: { id: true, name: true, slug: true },
        },
        images: {
          orderBy: { isPrimary: "desc" },
        },
        variants: {
          where: { stock: { gt: 0 } },
          orderBy: { price: "asc" },
        },
        attributes: {
          include: {
            attribute: true,
            attributeValue: true,
          },
        },
        tags: {
          include: {
            tag: true,
          },
        },
        reviews: {
          where: { isVerified: true },
          include: {
            user: {
              select: { name: true },
            },
          },
          orderBy: { createdAt: "desc" },
          take: 10,
        },
        _count: {
          select: { reviews: true },
        },
      },
    });

    if (!product) {
      return c.json({ error: "Product not found" }, 404);
    }

    // Calculate average rating
    const avgRating =
      product.reviews.length > 0
        ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
          product.reviews.length
        : 0;

    const { reviews, ...productWithoutReviews } = product;

    return c.json({
      ...productWithoutReviews,
      averageRating: Math.round(avgRating * 10) / 10,
      reviews,
    });
  } catch (error) {
    console.error("Get product by ID error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET RELATED PRODUCTS ------------------------- */
publicProductsRouter.get("/:id/related", async (c) => {
  try {
    const id = parseInt(c.req.param("id"));
    const prisma = c.get("prisma");

    if (isNaN(id)) {
      return c.json({ error: "Invalid product ID" }, 400);
    }

    const product = await prisma.product.findUnique({
      where: { id },
      select: { categoryId: true, tags: { include: { tag: true } } },
    });

    if (!product) {
      return c.json({ error: "Product not found" }, 404);
    }

    const relatedProducts = await prisma.product.findMany({
      where: {
        id: { not: id },
        status: "ACTIVE",
        OR: [
          { categoryId: product.categoryId },
          {
            tags: { some: { tagId: { in: product.tags.map((t) => t.tagId) } } },
          },
        ],
      },
      take: 8,
      include: {
        category: {
          select: { name: true, slug: true },
        },
        images: {
          where: { isPrimary: true },
          take: 1,
        },
        reviews: {
          select: { rating: true },
        },
      },
      orderBy: { createdAt: "desc" },
    });

    // Calculate average ratings
    const productsWithRating = relatedProducts.map((product) => {
      const avgRating =
        product.reviews.length > 0
          ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
            product.reviews.length
          : 0;

      const { reviews, ...productWithoutReviews } = product;
      return {
        ...productWithoutReviews,
        averageRating: Math.round(avgRating * 10) / 10,
      };
    });

    return c.json({ products: productsWithRating });
  } catch (error) {
    console.error("Get related products error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default publicProductsRouter;
