// routes/products.ts
import { requireAdmin, requireAuth } from "@/lib/auth.js";
import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { z } from "zod";

const productsRouter = new Hono<ContextVars>();

// Apply auth to all routes
productsRouter.use("*", requireAuth, withPrisma);

// Schemas
const createProductSchema = z.object({
  name: z.string().min(1),
  slug: z.string().min(1),
  description: z.string().optional(),
  sku: z.string().optional(),
  price: z.number().min(0),
  discountPrice: z.number().optional(),
  status: z
    .enum(["DRAFT", "ACTIVE", "INACTIVE", "OUT_OF_STOCK", "DISCONTINUED"])
    .default("DRAFT"),
  barcode: z.string().optional(),
  categoryId: z.number(),
  images: z
    .array(
      z.object({
        url: z.string().url(),
        altText: z.string().optional(),
        isPrimary: z.boolean().default(false),
      })
    )
    .optional(),
  variants: z
    .array(
      z.object({
        name: z.string().min(1),
        slug: z.string().min(1),
        sku: z.string().optional(),
        price: z.number().optional(),
        stock: z.number().default(0),
      })
    )
    .optional(),
  attributes: z
    .array(
      z.object({
        attributeId: z.number(),
        attributeValueId: z.number(),
      })
    )
    .optional(),
  tagIds: z.array(z.number()).optional(),
});

const updateProductSchema = createProductSchema.partial();

const productQuerySchema = z.object({
  page: z.string().transform(Number).optional().default(1),
  limit: z.string().transform(Number).optional().default(20),
  status: z.string().optional(),
  categoryId: z.string().transform(Number).optional(),
});

/* ------------------------- CREATE PRODUCT ------------------------- */
productsRouter.post(
  "/",
  requireAdmin,
  zValidator("json", createProductSchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const data = c.req.valid("json");

      // Check if slug already exists
      const existingProduct = await prisma.product.findUnique({
        where: { slug: data.slug },
      });

      if (existingProduct) {
        return c.json({ error: "Product slug already exists" }, 400);
      }

      const product = await prisma.product.create({
        data: {
          name: data.name,
          slug: data.slug,
          description: data.description,
          sku: data.sku,
          price: data.price,
          discountPrice: data.discountPrice,
          status: data.status,
          barcode: data.barcode,
          categoryId: data.categoryId,
          images: data.images
            ? {
                create: data.images,
              }
            : undefined,
          variants: data.variants
            ? {
                create: data.variants,
              }
            : undefined,
          tags: data.tagIds
            ? {
                create: data.tagIds.map((tagId) => ({ tagId })),
              }
            : undefined,
          attributes: data.attributes
            ? {
                create: data.attributes,
              }
            : undefined,
        },
        include: {
          category: true,
          images: true,
          variants: true,
          tags: { include: { tag: true } },
          attributes: {
            include: {
              attribute: true,
              attributeValue: true,
            },
          },
        },
      });

      return c.json({ product }, 201);
    } catch (error) {
      console.error("Create product error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- LIST PRODUCTS (Admin) ------------------------- */
productsRouter.get(
  "/",
  requireAdmin,
  zValidator("query", productQuerySchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const { page, limit, status, categoryId } = c.req.valid("query");
      const skip = (page - 1) * limit;

      const where: any = {};
      if (status) where.status = status;
      if (categoryId) where.categoryId = categoryId;

      const [products, total] = await Promise.all([
        prisma.product.findMany({
          where,
          skip,
          take: limit,
          include: {
            category: { select: { name: true } },
            images: { where: { isPrimary: true }, take: 1 },
            variants: true,
            _count: {
              select: { orders: true, reviews: true },
            },
          },
          orderBy: { createdAt: "desc" },
        }),
        prisma.product.count({ where }),
      ]);

      return c.json({
        products,
        pagination: {
          page,
          limit,
          total,
          pages: Math.ceil(total / limit),
        },
      });
    } catch (error) {
      console.error("Get products error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- UPDATE PRODUCT ------------------------- */
productsRouter.put(
  "/:id",
  requireAdmin,
  zValidator("json", updateProductSchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const id = parseInt(c.req.param("id"));
      const data = c.req.valid("json");

      if (isNaN(id)) {
        return c.json({ error: "Invalid product ID" }, 400);
      }

      const product = await prisma.product.update({
        where: { id },
        data: {
          name: data.name,
          slug: data.slug,
          description: data.description,
          sku: data.sku,
          price: data.price,
          discountPrice: data.discountPrice,
          status: data.status,
          barcode: data.barcode,
          categoryId: data.categoryId,
        },
        include: {
          category: true,
          images: true,
          variants: true,
        },
      });

      return c.json({ product });
    } catch (error) {
      console.error("Update product error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- DELETE PRODUCT ------------------------- */
productsRouter.delete("/:id", requireAdmin, async (c) => {
  try {
    const prisma = c.get("prisma");
    const id = parseInt(c.req.param("id"));

    if (isNaN(id)) {
      return c.json({ error: "Invalid product ID" }, 400);
    }

    await prisma.product.delete({
      where: { id },
    });

    return c.json({ message: "Product deleted successfully" });
  } catch (error) {
    console.error("Delete product error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- UPDATE INVENTORY ------------------------- */
productsRouter.patch(
  "/:id/inventory",
  requireAdmin,
  zValidator(
    "json",
    z.object({
      variantId: z.number().optional(),
      quantity: z.number(),
      movementType: z.enum([
        "PURCHASE",
        "SALE",
        "RETURN",
        "ADJUSTMENT",
        "TRANSFER",
      ]),
      reason: z.string().optional(),
    })
  ),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const productId = parseInt(c.req.param("id"));
      const { variantId, quantity, movementType, reason } = c.req.valid("json");

      if (isNaN(productId)) {
        return c.json({ error: "Invalid product ID" }, 400);
      }

      // Get current stock
      const product = variantId
        ? await prisma.productVariant.findUnique({
            where: { id: variantId },
            select: { stock: true, productId: true },
          })
        : await prisma.product.findUnique({
            where: { id: productId },
            select: { price: true }, // Using price as base stock indicator
          });

      if (!product) {
        return c.json({ error: "Product or variant not found" }, 404);
      }

      const previousStock = variantId ? (product as any).stock : 0;
      const newStock =
        movementType === "ADJUSTMENT" ? quantity : previousStock + quantity;

      // Update stock
      if (variantId) {
        await prisma.productVariant.update({
          where: { id: variantId },
          data: { stock: newStock },
        });
      }

      // Record inventory movement
      await prisma.inventoryMovement.create({
        data: {
          productId,
          variantId,
          movementType,
          quantity,
          previousStock,
          newStock,
          reason,
        },
      });

      return c.json({
        message: "Inventory updated successfully",
        previousStock,
        newStock,
      });
    } catch (error) {
      console.error("Update inventory error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

export default productsRouter;
