// routes/categories.ts
import { requireAdmin, requireAuth } from "@/lib/auth.js";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const categoriesRouter = new Hono<ContextVars>();

categoriesRouter.use("*", requireAuth, withPrisma);

// Schemas
const createCategorySchema = z.object({
  name: z.string().min(1),
  slug: z.string().min(1),
  description: z.string().optional(),
  image: z.string().optional(),
  parentId: z.number().optional(),
});

const updateCategorySchema = createCategorySchema.partial();

/* ------------------------- CREATE CATEGORY ------------------------- */
categoriesRouter.post(
  "/",
  requireAdmin,
  zValidator("json", createCategorySchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const data = c.req.valid("json");

      const existingCategory = await prisma.category.findUnique({
        where: { slug: data.slug },
      });

      if (existingCategory) {
        return c.json({ error: "Category slug already exists" }, 400);
      }

      const category = await prisma.category.create({
        data,
        include: {
          parent: true,
          children: true,
          _count: {
            select: { products: true },
          },
        },
      });

      return c.json({ category }, 201);
    } catch (error) {
      console.error("Create category error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- LIST CATEGORIES (Admin) ------------------------- */
categoriesRouter.get("/", requireAdmin, async (c) => {
  try {
    const prisma = c.get("prisma");
    const categories = await prisma.category.findMany({
      include: {
        parent: true,
        children: {
          include: {
            _count: {
              select: { products: true },
            },
          },
        },
        _count: {
          select: { products: true },
        },
      },
      orderBy: { name: "asc" },
    });

    return c.json({ categories });
  } catch (error) {
    console.error("Get categories error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- UPDATE CATEGORY ------------------------- */
categoriesRouter.put(
  "/:id",
  requireAdmin,
  zValidator("json", updateCategorySchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const id = parseInt(c.req.param("id"));
      const data = c.req.valid("json");

      if (isNaN(id)) {
        return c.json({ error: "Invalid category ID" }, 400);
      }

      const category = await prisma.category.update({
        where: { id },
        data,
        include: {
          parent: true,
          children: true,
          _count: {
            select: { products: true },
          },
        },
      });

      return c.json({ category });
    } catch (error) {
      console.error("Update category error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- DELETE CATEGORY ------------------------- */
categoriesRouter.delete("/:id", requireAdmin, async (c) => {
  try {
    const prisma = c.get("prisma");
    const id = parseInt(c.req.param("id"));

    if (isNaN(id)) {
      return c.json({ error: "Invalid category ID" }, 400);
    }

    // Check if category has products
    const categoryWithProducts = await prisma.category.findUnique({
      where: { id },
      include: {
        _count: {
          select: { products: true, children: true },
        },
      },
    });

    if (categoryWithProducts?._count.products > 0) {
      return c.json({ error: "Cannot delete category with products" }, 400);
    }

    if (categoryWithProducts?._count.children > 0) {
      return c.json(
        { error: "Cannot delete category with subcategories" },
        400
      );
    }

    await prisma.category.delete({
      where: { id },
    });

    return c.json({ message: "Category deleted successfully" });
  } catch (error) {
    console.error("Delete category error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default categoriesRouter;
