import {
  authMiddleware,
  generateToken,
  hashPassword,
  jwtMiddleware,
  requireAuth,
  verifyPassword,
} from "@/lib/auth.js";
import {
  forgotPasswordSchema,
  loginSchema,
  registerSchema,
  resetPasswordSchema,
  updateProfileSchema,
} from "@/lib/schema.js";

import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const usersRouter = new Hono<ContextVars>();

/* --------------------- APPLY AUTH MIDDLEWARE --------------------- */
// Apply JWT middleware first, then auth middleware
usersRouter.use("*", jwtMiddleware, authMiddleware, withPrisma);

/* ------------------------- REGISTER ------------------------- */
usersRouter.post("/register", zValidator("json", registerSchema), async (c) => {
  try {
    const prisma = c.get("prisma");
    const { email, password, name, phone } = c.req.valid("json");

    const existingUser = await prisma.user.findUnique({ where: { email } });
    if (existingUser) return c.json({ error: "User already exists" }, 400);

    const hashedPassword = await hashPassword(password);
    const user = await prisma.user.create({
      data: { email, password: hashedPassword, name, phone, role: "USER" },
    });

    const token = await generateToken({
      id: user.id,
      email: user.email,
      role: user.role,
    });

    const { password: _, ...userWithoutPassword } = user;
    return c.json({ user: userWithoutPassword, token }, 201);
  } catch (error) {
    console.error("Registration error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- LOGIN ------------------------- */
usersRouter.post("/login", zValidator("json", loginSchema), async (c) => {
  try {
    const prisma = c.get("prisma");
    const { email, password } = c.req.valid("json");

    const user = await prisma.user.findUnique({ where: { email } });
    if (!user || !(await verifyPassword(password, user.password)))
      return c.json({ error: "Invalid credentials" }, 401);

    const token = await generateToken({
      id: user.id,
      email: user.email,
      role: user.role,
    });

    await prisma.session.create({
      data: {
        userId: user.id,
        expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000),
      },
    });

    const { password: _, ...userWithoutPassword } = user;
    return c.json({ user: userWithoutPassword, token });
  } catch (error) {
    console.error("Login error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET PROFILE ------------------------- */
usersRouter.get("/profile", requireAuth, async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");

  const userData = await prisma.user.findUnique({
    where: { id: user.id },
    select: {
      id: true,
      email: true,
      name: true,
      phone: true,
      role: true,
      createdAt: true,
      updatedAt: true,
    },
  });

  return c.json({ user: userData });
});

/* ------------------------- UPDATE PROFILE ------------------------- */
usersRouter.patch(
  "/profile",
  requireAuth,
  zValidator("json", updateProfileSchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const user = c.get("user");
      const updates = c.req.valid("json");

      const updatedUser = await prisma.user.update({
        where: { id: user.id },
        data: updates,
        select: {
          id: true,
          email: true,
          name: true,
          phone: true,
          createdAt: true,
          updatedAt: true,
        },
      });

      return c.json({ user: updatedUser });
    } catch (error) {
      console.error("Profile update error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- LOGOUT ------------------------- */
usersRouter.post("/logout", requireAuth, async (c) => {
  try {
    const prisma = c.get("prisma");
    const user = c.get("user");
    // Delete all sessions for the user
    await prisma.session.deleteMany({
      where: { userId: user.id },
    });

    return c.json({ message: "Logged out successfully" });
  } catch (error) {
    console.error("Logout error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- FORGOT PASSWORD ------------------------- */
usersRouter.post(
  "/forgot-password",
  zValidator("json", forgotPasswordSchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const { email } = c.req.valid("json");

      const user = await prisma.user.findUnique({ where: { email } });
      if (user) {
        const resetToken = await prisma.passwordResetToken.create({
          data: {
            userId: user.id,
            expiresAt: new Date(Date.now() + 1 * 60 * 60 * 1000),
          },
        });
        console.log(`Reset token for ${email}: ${resetToken.id}`);
      }

      return c.json({
        message:
          "If an account with that email exists, a reset link has been sent",
      });
    } catch (error) {
      console.error("Forgot password error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- RESET PASSWORD ------------------------- */
usersRouter.post(
  "/reset-password",
  zValidator("json", resetPasswordSchema),
  async (c) => {
    try {
      const prisma = c.get("prisma");
      const { token, newPassword } = c.req.valid("json");

      const resetToken = await prisma.passwordResetToken.findFirst({
        where: {
          id: token,
          used: false,
          expiresAt: { gt: new Date() },
        },
        include: { user: true },
      });

      if (!resetToken)
        return c.json({ error: "Invalid or expired reset token" }, 400);

      const hashedPassword = await hashPassword(newPassword);
      await prisma.user.update({
        where: { id: resetToken.userId },
        data: { password: hashedPassword },
      });

      await prisma.passwordResetToken.update({
        where: { id: token },
        data: { used: true },
      });

      await prisma.session.deleteMany({
        where: { userId: resetToken.userId },
      });

      return c.json({ message: "Password reset successfully" });
    } catch (error) {
      console.error("Reset password error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

// /* ------------------------- ADMIN ROUTES ------------------------- */
// usersRouter.get("/admin/users", requireAdmin, async (c) => {
//   try {
//     const users = await prisma.user.findMany({
//       select: {
//         id: true,
//         email: true,
//         name: true,
//         phone: true,
//         role: true,
//         createdAt: true,
//         updatedAt: true,
//       },
//     });

//     return c.json({ users });
//   } catch (error) {
//     console.error("Admin users list error:", error);
//     return c.json({ error: "Internal server error" }, 500);
//   }
// });

// usersRouter.patch(
//   "/admin/users/:id/role",
//   requireAdmin,
//   zValidator("json", updateRoleSchema),
//   async (c) => {
//     try {
//       const userId = parseInt(c.req.param("id"));
//       const { role } = c.req.valid("json");

//       const updatedUser = await prisma.user.update({
//         where: { id: userId },
//         data: { role },
//         select: { id: true, email: true, name: true, role: true },
//       });

//       return c.json({ user: updatedUser });
//     } catch (error) {
//       console.error("Update user role error:", error);
//       return c.json({ error: "Internal server error" }, 500);
//     }
//   }
// );

export default usersRouter;
