// routes/addresses.ts
import { Hono } from "hono";
import { zValidator } from "@hono/zod-validator";
import { requireAuth } from "@/lib/auth.js";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const addressesRouter = new Hono<ContextVars>();

addressesRouter.use("*", requireAuth);

// Schemas
const createAddressSchema = z.object({
  firstName: z.string().min(1),
  lastName: z.string().min(1),
  company: z.string().optional(),
  addressLine1: z.string().min(1),
  addressLine2: z.string().optional(),
  city: z.string().min(1),
  state: z.string().min(1),
  postalCode: z.string().min(1),
  country: z.string().default("US"),
  phone: z.string().optional(),
  isDefault: z.boolean().default(false),
});

const updateAddressSchema = createAddressSchema.partial();

addressesRouter.use("*", withPrisma);

/* ------------------------- GET ADDRESSES ------------------------- */
addressesRouter.get("/", async (c) => {
  const user = c.get("user");
  const prisma = c.get("prisma");

  try {
    const addresses = await prisma.address.findMany({
      where: { userId: user.id },
      orderBy: [{ isDefault: "desc" }, { createdAt: "desc" }],
    });

    return c.json({ addresses });
  } catch (error) {
    console.error("Get addresses error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- CREATE ADDRESS ------------------------- */
addressesRouter.post(
  "/",
  zValidator("json", createAddressSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const user = c.get("user");
    const data = c.req.valid("json");

    try {
      // If setting as default, unset other defaults
      if (data.isDefault) {
        await prisma.address.updateMany({
          where: { userId: user.id, isDefault: true },
          data: { isDefault: false },
        });
      }

      const address = await prisma.address.create({
        data: {
          ...data,
          userId: user.id,
        },
      });

      return c.json({ address }, 201);
    } catch (error) {
      console.error("Create address error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- UPDATE ADDRESS ------------------------- */
addressesRouter.put(
  "/:id",
  zValidator("json", updateAddressSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const user = c.get("user");
    const addressId = parseInt(c.req.param("id"));
    const data = c.req.valid("json");

    if (isNaN(addressId)) {
      return c.json({ error: "Invalid address ID" }, 400);
    }

    try {
      // Verify user owns the address
      const address = await prisma.address.findFirst({
        where: {
          id: addressId,
          userId: user.id,
        },
      });

      if (!address) {
        return c.json({ error: "Address not found" }, 404);
      }

      // If setting as default, unset other defaults
      if (data.isDefault) {
        await prisma.address.updateMany({
          where: { userId: user.id, isDefault: true },
          data: { isDefault: false },
        });
      }

      const updatedAddress = await prisma.address.update({
        where: { id: addressId },
        data,
      });

      return c.json({ address: updatedAddress });
    } catch (error) {
      console.error("Update address error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- DELETE ADDRESS ------------------------- */
addressesRouter.delete("/:id", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const addressId = parseInt(c.req.param("id"));

  if (isNaN(addressId)) {
    return c.json({ error: "Invalid address ID" }, 400);
  }

  try {
    // Verify user owns the address
    const address = await prisma.address.findFirst({
      where: {
        id: addressId,
        userId: user.id,
      },
    });

    if (!address) {
      return c.json({ error: "Address not found" }, 404);
    }

    await prisma.address.delete({
      where: { id: addressId },
    });

    return c.json({ message: "Address deleted successfully" });
  } catch (error) {
    console.error("Delete address error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- SET DEFAULT ADDRESS ------------------------- */
addressesRouter.patch("/:id/default", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const addressId = parseInt(c.req.param("id"));

  if (isNaN(addressId)) {
    return c.json({ error: "Invalid address ID" }, 400);
  }

  try {
    // Verify user owns the address
    const address = await prisma.address.findFirst({
      where: {
        id: addressId,
        userId: user.id,
      },
    });

    if (!address) {
      return c.json({ error: "Address not found" }, 404);
    }

    // Unset all other defaults
    await prisma.address.updateMany({
      where: { userId: user.id, isDefault: true },
      data: { isDefault: false },
    });

    // Set this as default
    const updatedAddress = await prisma.address.update({
      where: { id: addressId },
      data: { isDefault: true },
    });

    return c.json({ address: updatedAddress });
  } catch (error) {
    console.error("Set default address error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default addressesRouter;
