import { serve } from "@hono/node-server";
import { Hono } from "hono";
import { cors } from "hono/cors";
import { showRoutes } from "hono/dev";
import { logger } from "hono/logger";
// Import routers
import usersRouter from "./routes/auth/auth.js";
import docRouter from "./docs.js";
import publicRouter from "./routes/public/index.js";
import addressesRouter from "./routes/addresses.js";
import basketRouter from "./routes/basket.js";
import categoriesRouter from "./routes/categories.js";
import ordersRouter from "./routes/orders.js";
import paymentsRouter from "./routes/payments.js";
import productsRouter from "./routes/products.js";
import reviewsRouter from "./routes/reviews.js";
import wishlistRouter from "./routes/wishlist.js";
import type { ContextVars } from "@/types/contextVars.js";

const app = new Hono<ContextVars>();

// Add CORS middleware
app.use("*", cors(), logger());

// Health check
app.get("/health", (c) =>
  c.json({ status: "OK", timestamp: new Date().toISOString() })
);

// Public routes (no auth required)
app.route("/public", publicRouter);

// Authentication routes
app.route("/auth", usersRouter);

// Protected routes (require authentication)
app.route("/products", productsRouter);
app.route("/categories", categoriesRouter);
app.route("/basket", basketRouter);
app.route("/orders", ordersRouter);
app.route("/payments", paymentsRouter);
app.route("/reviews", reviewsRouter);
app.route("/wishlist", wishlistRouter);
app.route("/addresses", addressesRouter);

// Documentation routes
app.route("/docs", docRouter);

// 404 handler
app.notFound((c) => c.json({ error: "Route not found" }, 404));

// Global error handler
app.onError((err, c) => {
  console.error("Server error:", err);
  return c.json({ error: "Internal server error" }, 500);
});

// Show routes in development
showRoutes(app);

// Start server
serve(
  {
    fetch: app.fetch,
    port: 4000,
  },
  (info) => {
    console.log(`🚀 Server is running on http://localhost:${info.port}`);
    console.log(`📚 API Documentation: http://localhost:${info.port}/docs/ui`);
    console.log(`📋 API Spec: http://localhost:${info.port}/docs/doc`);
    console.log(`❤️  Health check: http://localhost:${info.port}/health`);
  }
);

export default app;
