// routes/wishlist.ts
import { Hono } from "hono";
import { zValidator } from "@hono/zod-validator";
import { requireAuth } from "@/lib/auth.js";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const wishlistRouter = new Hono<ContextVars>();

wishlistRouter.use("*", requireAuth);

// Schemas
const addToWishlistSchema = z.object({
  productId: z.number(),
});

wishlistRouter.use("*", withPrisma);

/* ------------------------- GET WISHLIST ------------------------- */
wishlistRouter.get("/", async (c) => {
  try {
    const user = c.get("user");
    const prisma = c.get("prisma");

    let wishlist = await prisma.wishlist.findFirst({
      where: { userId: user.id },
      include: {
        items: {
          include: {
            product: {
              include: {
                images: {
                  where: { isPrimary: true },
                  take: 1,
                },
                variants: {
                  where: { stock: { gt: 0 } },
                  take: 1,
                },
                reviews: {
                  select: { rating: true },
                },
                _count: {
                  select: { reviews: true },
                },
              },
            },
          },
        },
      },
    });

    // Create wishlist if it doesn't exist
    if (!wishlist) {
      wishlist = await prisma.wishlist.create({
        data: { userId: user.id },
        include: {
          items: {
            include: {
              product: {
                include: {
                  images: {
                    where: { isPrimary: true },
                    take: 1,
                  },
                  variants: {
                    where: { stock: { gt: 0 } },
                    take: 1,
                  },
                  reviews: {
                    select: { rating: true },
                  },
                  _count: {
                    select: { reviews: true },
                  },
                },
              },
            },
          },
        },
      });
    }

    // Calculate average ratings
    const itemsWithRating = wishlist.items.map((item) => {
      const avgRating =
        item.product.reviews.length > 0
          ? item.product.reviews.reduce(
              (sum, review) => sum + review.rating,
              0
            ) / item.product.reviews.length
          : 0;

      const { reviews, ...productWithoutReviews } = item.product;
      return {
        ...item,
        product: {
          ...productWithoutReviews,
          averageRating: Math.round(avgRating * 10) / 10,
          reviewCount: item.product._count.reviews,
        },
      };
    });

    return c.json({
      wishlist: {
        ...wishlist,
        items: itemsWithRating,
      },
    });
  } catch (error) {
    console.error("Get wishlist error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- ADD TO WISHLIST ------------------------- */
wishlistRouter.post(
  "/items",
  zValidator("json", addToWishlistSchema),
  async (c) => {
    const user = c.get("user");
    const { productId } = c.req.valid("json");
    const prisma = c.get("prisma");
    try {
      // Get or create wishlist
      let wishlist = await prisma.wishlist.findFirst({
        where: { userId: user.id },
      });

      if (!wishlist) {
        wishlist = await prisma.wishlist.create({
          data: { userId: user.id },
        });
      }

      // Check if item already in wishlist
      const existingItem = await prisma.wishlistItem.findUnique({
        where: {
          wishlistId_productId: {
            wishlistId: wishlist.id,
            productId,
          },
        },
      });

      if (existingItem) {
        return c.json({ error: "Product already in wishlist" }, 400);
      }

      const wishlistItem = await prisma.wishlistItem.create({
        data: {
          wishlistId: wishlist.id,
          productId,
        },
        include: {
          product: {
            include: {
              images: {
                where: { isPrimary: true },
                take: 1,
              },
            },
          },
        },
      });

      return c.json({ item: wishlistItem }, 201);
    } catch (error) {
      console.error("Add to wishlist error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- REMOVE FROM WISHLIST ------------------------- */
wishlistRouter.delete("/items/:productId", async (c) => {
  const user = c.get("user");
  const productId = parseInt(c.req.param("productId"));
  const prisma = c.get("prisma");

  if (isNaN(productId)) {
    return c.json({ error: "Invalid product ID" }, 400);
  }

  try {
    const wishlist = await prisma.wishlist.findFirst({
      where: { userId: user.id },
    });

    if (!wishlist) {
      return c.json({ error: "Wishlist not found" }, 404);
    }

    await prisma.wishlistItem.delete({
      where: {
        wishlistId_productId: {
          wishlistId: wishlist.id,
          productId,
        },
      },
    });

    return c.json({ message: "Item removed from wishlist" });
  } catch (error) {
    console.error("Remove from wishlist error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- CLEAR WISHLIST ------------------------- */
wishlistRouter.delete("/", async (c) => {
  const user = c.get("user");
  const prisma = c.get("prisma");

  try {
    const wishlist = await prisma.wishlist.findFirst({
      where: { userId: user.id },
    });

    if (wishlist) {
      await prisma.wishlistItem.deleteMany({
        where: { wishlistId: wishlist.id },
      });
    }

    return c.json({ message: "Wishlist cleared successfully" });
  } catch (error) {
    console.error("Clear wishlist error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default wishlistRouter;
