// routes/public/categories.ts
import { Hono } from "hono";
import type { ContextVars } from "@/types/contextVars.js";
import withPrisma from "@/lib/prisma.js";

const publicCategoriesRouter = new Hono<ContextVars>();

publicCategoriesRouter.use("*", withPrisma);

/* ------------------------- LIST CATEGORIES ------------------------- */
publicCategoriesRouter.get("/", async (c) => {
  try {
    const prisma = c.get("prisma");

    const categories = await prisma.category.findMany({
      where: {
        parentId: null, // Only get top-level categories
      },
      include: {
        children: {
          include: {
            _count: {
              select: { products: true },
            },
          },
        },
        _count: {
          select: { products: true },
        },
      },
      orderBy: { name: "asc" },
    });

    return c.json({ categories });
  } catch (error) {
    console.error("Get categories error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET CATEGORY BY SLUG ------------------------- */
publicCategoriesRouter.get("/slug/:slug", async (c) => {
  try {
    const prisma = c.get("prisma");

    const slug = c.req.param("slug");

    const category = await prisma.category.findUnique({
      where: { slug },
      include: {
        parent: true,
        children: {
          include: {
            _count: {
              select: { products: true },
            },
          },
        },
        products: {
          where: { status: "ACTIVE" },
          include: {
            images: {
              where: { isPrimary: true },
              take: 1,
            },
            reviews: {
              select: { rating: true },
            },
          },
          take: 20, // Limit products in category response
        },
        _count: {
          select: { products: true },
        },
      },
    });

    if (!category) {
      return c.json({ error: "Category not found" }, 404);
    }

    // Calculate average ratings for products
    const productsWithRating = category.products.map((product) => {
      const avgRating =
        product.reviews.length > 0
          ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
            product.reviews.length
          : 0;

      const { reviews, ...productWithoutReviews } = product;
      return {
        ...productWithoutReviews,
        averageRating: Math.round(avgRating * 10) / 10,
      };
    });

    return c.json({
      ...category,
      products: productsWithRating,
    });
  } catch (error) {
    console.error("Get category by slug error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET CATEGORY PRODUCTS ------------------------- */
publicCategoriesRouter.get("/:id/products", async (c) => {
  try {
    const prisma = c.get("prisma");

    const id = parseInt(c.req.param("id"));
    const page = parseInt(c.req.query("page") || "1");
    const limit = parseInt(c.req.query("limit") || "20");

    if (isNaN(id)) {
      return c.json({ error: "Invalid category ID" }, 400);
    }

    const skip = (page - 1) * limit;

    // Get category and its subcategories
    const category = await prisma.category.findUnique({
      where: { id },
      include: {
        children: {
          select: { id: true },
        },
      },
    });

    if (!category) {
      return c.json({ error: "Category not found" }, 404);
    }

    // Get all category IDs (parent + children)
    const categoryIds = [
      category.id,
      ...category.children.map((child) => child.id),
    ];

    const [products, total] = await Promise.all([
      prisma.product.findMany({
        where: {
          categoryId: { in: categoryIds },
          status: "ACTIVE",
        },
        skip,
        take: limit,
        include: {
          images: {
            where: { isPrimary: true },
            take: 1,
          },
          reviews: {
            select: { rating: true },
          },
          category: {
            select: { name: true, slug: true },
          },
        },
        orderBy: { createdAt: "desc" },
      }),
      prisma.product.count({
        where: {
          categoryId: { in: categoryIds },
          status: "ACTIVE",
        },
      }),
    ]);

    // Calculate average ratings
    const productsWithRating = products.map((product) => {
      const avgRating =
        product.reviews.length > 0
          ? product.reviews.reduce((sum, review) => sum + review.rating, 0) /
            product.reviews.length
          : 0;

      const { reviews, ...productWithoutReviews } = product;
      return {
        ...productWithoutReviews,
        averageRating: Math.round(avgRating * 10) / 10,
      };
    });

    return c.json({
      category: {
        id: category.id,
        name: category.name,
        slug: category.slug,
      },
      products: productsWithRating,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error("Get category products error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default publicCategoriesRouter;
