// routes/payments.ts
import { requireAdmin, requireAuth } from "@/lib/auth.js";
import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { z } from "zod";

const paymentsRouter = new Hono<ContextVars>();

paymentsRouter.use("*", requireAuth, withPrisma);

// Schemas
const createPaymentIntentSchema = z.object({
  orderId: z.string().min(1),
  paymentMethod: z.string().min(1),
});

const confirmPaymentSchema = z.object({
  paymentIntentId: z.string().min(1),
  transactionId: z.string().optional(),
  metadata: z.record(z.any()).optional(),
});

/* ------------------------- CREATE PAYMENT INTENT ------------------------- */
paymentsRouter.post(
  "/intent",
  zValidator("json", createPaymentIntentSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const user = c.get("user");
    const { orderId, paymentMethod } = c.req.valid("json");

    try {
      // Verify order belongs to user
      const order = await prisma.order.findFirst({
        where: {
          id: orderId,
          userId: user.id,
        },
        include: {
          payments: true,
        },
      });

      if (!order) {
        return c.json({ error: "Order not found" }, 404);
      }

      // Check if payment already exists
      const existingPayment = order.payments.find(
        (p) => p.status === "PENDING"
      );
      if (existingPayment) {
        return c.json({
          paymentIntentId: existingPayment.paymentIntentId,
          clientSecret: `cs_${existingPayment.id}`, // Mock client secret
        });
      }

      // Create payment record
      const payment = await prisma.payment.create({
        data: {
          orderId,
          userId: user.id,
          paymentMethod,
          amount: order.finalAmount,
          status: "PENDING",
          paymentIntentId: `pi_${Date.now()}_${Math.random()
            .toString(36)
            .substr(2, 9)}`,
        },
      });

      // In a real implementation, you would integrate with Stripe/PayPal here
      return c.json({
        paymentIntentId: payment.paymentIntentId,
        clientSecret: `cs_${payment.id}`, // Mock client secret
        amount: payment.amount,
        currency: payment.currency,
      });
    } catch (error) {
      console.error("Create payment intent error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- CONFIRM PAYMENT ------------------------- */
paymentsRouter.post(
  "/confirm",
  zValidator("json", confirmPaymentSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const user = c.get("user");
    const { paymentIntentId, transactionId, metadata } = c.req.valid("json");

    try {
      // Find payment
      const payment = await prisma.payment.findFirst({
        where: {
          paymentIntentId,
          userId: user.id,
        },
        include: {
          order: true,
        },
      });

      if (!payment) {
        return c.json({ error: "Payment not found" }, 404);
      }

      // Update payment status
      const updatedPayment = await prisma.payment.update({
        where: { id: payment.id },
        data: {
          status: "COMPLETED",
          transactionId,
          metadata: metadata as any,
          paidAt: new Date(),
        },
      });

      // Update order status
      await prisma.order.update({
        where: { id: payment.orderId },
        data: { status: "CONFIRMED" },
      });

      return c.json({
        payment: updatedPayment,
        message: "Payment confirmed successfully",
      });
    } catch (error) {
      console.error("Confirm payment error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- GET PAYMENT STATUS ------------------------- */
paymentsRouter.get("/:id", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const paymentId = c.req.param("id");

  try {
    const payment = await prisma.payment.findFirst({
      where: {
        id: paymentId,
        userId: user.id,
      },
      include: {
        order: {
          select: {
            id: true,
            orderNumber: true,
            status: true,
          },
        },
      },
    });

    if (!payment) {
      return c.json({ error: "Payment not found" }, 404);
    }

    return c.json({ payment });
  } catch (error) {
    console.error("Get payment error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- ADMIN: REFUND PAYMENT ------------------------- */
paymentsRouter.post("/admin/:id/refund", requireAdmin, async (c) => {
  const prisma = c.get("prisma");
  const paymentId = c.req.param("id");

  try {
    const payment = await prisma.payment.findUnique({
      where: { id: paymentId },
      include: { order: true },
    });

    if (!payment) {
      return c.json({ error: "Payment not found" }, 404);
    }

    if (payment.status !== "COMPLETED") {
      return c.json({ error: "Only completed payments can be refunded" }, 400);
    }

    // Update payment status
    const updatedPayment = await prisma.payment.update({
      where: { id: paymentId },
      data: { status: "REFUNDED" },
    });

    // Update order status
    await prisma.order.update({
      where: { id: payment.orderId },
      data: { status: "REFUNDED" },
    });

    return c.json({
      payment: updatedPayment,
      message: "Payment refunded successfully",
    });
  } catch (error) {
    console.error("Refund payment error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default paymentsRouter;
