// routes/orders.ts
import { Hono } from "hono";
import { zValidator } from "@hono/zod-validator";
import { requireAuth, requireAdmin } from "@/lib/auth.js";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const ordersRouter = new Hono<ContextVars>();

ordersRouter.use("*", requireAuth, withPrisma);

// Schemas
const createOrderSchema = z.object({
  shippingAddressId: z.number(),
  billingAddressId: z.number().optional(),
  paymentMethod: z.string().min(1),
  notes: z.string().optional(),
});

const updateOrderStatusSchema = z.object({
  status: z.enum([
    "PENDING",
    "CONFIRMED",
    "PROCESSING",
    "SHIPPED",
    "DELIVERED",
    "CANCELLED",
    "REFUNDED",
  ]),
});

/* ------------------------- CREATE ORDER ------------------------- */
ordersRouter.post("/", zValidator("json", createOrderSchema), async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const { shippingAddressId, billingAddressId, paymentMethod, notes } =
    c.req.valid("json");

  try {
    // Get user's basket
    const basket = await prisma.basket.findFirst({
      where: { userId: user.id },
      include: {
        items: {
          include: {
            product: true,
            variant: true,
          },
        },
      },
    });

    if (!basket || basket.items.length === 0) {
      return c.json({ error: "Basket is empty" }, 400);
    }

    // Calculate totals
    let totalAmount = 0;
    const orderItems = basket.items.map((item) => {
      const price = item.variant?.price || item.product.price;
      const itemTotal = price * item.quantity;
      totalAmount += itemTotal;

      return {
        productId: item.productId,
        variantId: item.variantId,
        quantity: item.quantity,
        unitPrice: price,
        totalPrice: itemTotal,
      };
    });

    const finalAmount = totalAmount; // Add shipping, tax, discount calculations

    // Generate order number
    const orderNumber = `ORD-${Date.now()}-${Math.random()
      .toString(36)
      .substr(2, 9)}`;

    // Create order
    const order = await prisma.order.create({
      data: {
        orderNumber,
        userId: user.id,
        status: "PENDING",
        totalAmount,
        finalAmount,
        shippingAddressId,
        billingAddressId: billingAddressId || shippingAddressId,
        items: {
          create: orderItems,
        },
        payments: {
          create: {
            userId: user.id,
            paymentMethod,
            amount: finalAmount,
            status: "PENDING",
          },
        },
      },
      include: {
        items: {
          include: {
            product: {
              include: {
                images: {
                  where: { isPrimary: true },
                  take: 1,
                },
              },
            },
            variant: true,
          },
        },
        shippingAddress: true,
        billingAddress: true,
        payments: true,
      },
    });

    // Clear basket after order creation
    await prisma.basketItem.deleteMany({
      where: { basketId: basket.id },
    });

    return c.json({ order }, 201);
  } catch (error) {
    console.error("Create order error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET USER ORDERS ------------------------- */
ordersRouter.get("/", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const page = parseInt(c.req.query("page") || "1");
  const limit = parseInt(c.req.query("limit") || "10");

  try {
    const skip = (page - 1) * limit;

    const [orders, total] = await Promise.all([
      prisma.order.findMany({
        where: { userId: user.id },
        skip,
        take: limit,
        include: {
          items: {
            include: {
              product: {
                include: {
                  images: {
                    where: { isPrimary: true },
                    take: 1,
                  },
                },
              },
            },
          },
          payments: {
            select: {
              status: true,
              paymentMethod: true,
              amount: true,
            },
          },
        },
        orderBy: { createdAt: "desc" },
      }),
      prisma.order.count({
        where: { userId: user.id },
      }),
    ]);

    return c.json({
      orders,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error("Get orders error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- GET ORDER DETAILS ------------------------- */
ordersRouter.get("/:id", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const orderId = c.req.param("id");

  try {
    const order = await prisma.order.findFirst({
      where: {
        id: orderId,
        userId: user.id,
      },
      include: {
        items: {
          include: {
            product: {
              include: {
                images: true,
              },
            },
            variant: true,
          },
        },
        shippingAddress: true,
        billingAddress: true,
        payments: true,
      },
    });

    if (!order) {
      return c.json({ error: "Order not found" }, 404);
    }

    return c.json({ order });
  } catch (error) {
    console.error("Get order error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- CANCEL ORDER ------------------------- */
ordersRouter.post("/:id/cancel", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const orderId = c.req.param("id");

  try {
    const order = await prisma.order.findFirst({
      where: {
        id: orderId,
        userId: user.id,
      },
    });

    if (!order) {
      return c.json({ error: "Order not found" }, 404);
    }

    // Only allow cancellation for pending/confirmed orders
    if (!["PENDING", "CONFIRMED"].includes(order.status)) {
      return c.json({ error: "Order cannot be cancelled at this stage" }, 400);
    }

    const updatedOrder = await prisma.order.update({
      where: { id: orderId },
      data: { status: "CANCELLED" },
      include: {
        items: true,
        payments: true,
      },
    });

    return c.json({ order: updatedOrder });
  } catch (error) {
    console.error("Cancel order error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- ADMIN: GET ALL ORDERS ------------------------- */
ordersRouter.get("/admin/orders", requireAdmin, async (c) => {
  const prisma = c.get("prisma");
  const page = parseInt(c.req.query("page") || "1");
  const limit = parseInt(c.req.query("limit") || "20");
  const status = c.req.query("status");

  try {
    const skip = (page - 1) * limit;

    const where: any = {};
    if (status) where.status = status;

    const [orders, total] = await Promise.all([
      prisma.order.findMany({
        where,
        skip,
        take: limit,
        include: {
          user: {
            select: { name: true, email: true },
          },
          items: {
            include: {
              product: {
                select: { name: true },
              },
            },
          },
          payments: {
            select: {
              status: true,
              paymentMethod: true,
            },
          },
        },
        orderBy: { createdAt: "desc" },
      }),
      prisma.order.count({ where }),
    ]);

    return c.json({
      orders,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error("Admin get orders error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- ADMIN: UPDATE ORDER STATUS ------------------------- */
ordersRouter.patch(
  "/admin/orders/:id/status",
  requireAdmin,
  zValidator("json", updateOrderStatusSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const orderId = c.req.param("id");
    const { status } = c.req.valid("json");

    try {
      const order = await prisma.order.update({
        where: { id: orderId },
        data: { status },
        include: {
          user: {
            select: { name: true, email: true },
          },
          items: true,
        },
      });

      return c.json({ order });
    } catch (error) {
      console.error("Update order status error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

export default ordersRouter;
