// routes/basket.ts
import { requireAuth } from "@/lib/auth.js";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { z } from "zod";

import withPrisma from "@/lib/prisma.js";
import type { ContextVars } from "@/types/contextVars.js";

const basketRouter = new Hono<ContextVars>();

basketRouter.use("*", requireAuth);

// Schemas
const addItemSchema = z.object({
  productId: z.number(),
  variantId: z.number().optional(),
  quantity: z.number().min(1).default(1),
});

const updateItemSchema = z.object({
  quantity: z.number().min(0), // 0 means remove
});

basketRouter.use("*", withPrisma);
/* ------------------------- GET BASKET ------------------------- */
basketRouter.get("/", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");

  try {
    let basket = await prisma.basket.findFirst({
      where: { userId: user.id },
      include: {
        items: {
          include: {
            product: {
              include: {
                images: {
                  where: { isPrimary: true },
                  take: 1,
                },
                variants: true,
              },
            },
            variant: true,
          },
        },
      },
    });

    // Create basket if it doesn't exist
    if (!basket) {
      basket = await prisma.basket.create({
        data: { userId: user.id },
        include: {
          items: {
            include: {
              product: {
                include: {
                  images: {
                    where: { isPrimary: true },
                    take: 1,
                  },
                  variants: true,
                },
              },
              variant: true,
            },
          },
        },
      });
    }

    return c.json({ basket });
  } catch (error) {
    console.error("Get basket error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- ADD ITEM TO BASKET ------------------------- */
basketRouter.post("/items", zValidator("json", addItemSchema), async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const { productId, variantId, quantity } = c.req.valid("json");

  try {
    // Get or create basket
    let basket = await prisma.basket.findFirst({
      where: { userId: user.id },
    });

    if (!basket) {
      basket = await prisma.basket.create({
        data: { userId: user.id },
      });
    }

    // Check if item already exists
    const existingItem = await prisma.basketItem.findFirst({
      where: {
        basketId: basket.id,
        productId,
        variantId: variantId || null,
      },
    });

    let basketItem;
    if (existingItem) {
      // Update quantity
      basketItem = await prisma.basketItem.update({
        where: { id: existingItem.id },
        data: { quantity: existingItem.quantity + quantity },
        include: {
          product: {
            include: {
              images: {
                where: { isPrimary: true },
                take: 1,
              },
            },
          },
          variant: true,
        },
      });
    } else {
      // Create new item
      basketItem = await prisma.basketItem.create({
        data: {
          basketId: basket.id,
          productId,
          variantId,
          quantity,
        },
        include: {
          product: {
            include: {
              images: {
                where: { isPrimary: true },
                take: 1,
              },
            },
          },
          variant: true,
        },
      });
    }

    return c.json({ item: basketItem }, 201);
  } catch (error) {
    console.error("Add to basket error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- UPDATE BASKET ITEM ------------------------- */
basketRouter.put(
  "/items/:itemId",
  zValidator("json", updateItemSchema),
  async (c) => {
    const prisma = c.get("prisma");
    const user = c.get("user");
    const itemId = c.req.param("itemId");
    const { quantity } = c.req.valid("json");

    try {
      // Verify user owns this basket item
      const basketItem = await prisma.basketItem.findFirst({
        where: {
          id: itemId,
          basket: { userId: user.id },
        },
      });

      if (!basketItem) {
        return c.json({ error: "Basket item not found" }, 404);
      }

      if (quantity === 0) {
        // Remove item
        await prisma.basketItem.delete({
          where: { id: itemId },
        });
        return c.json({ message: "Item removed from basket" });
      }

      // Update quantity
      const updatedItem = await prisma.basketItem.update({
        where: { id: itemId },
        data: { quantity },
        include: {
          product: {
            include: {
              images: {
                where: { isPrimary: true },
                take: 1,
              },
            },
          },
          variant: true,
        },
      });

      return c.json({ item: updatedItem });
    } catch (error) {
      console.error("Update basket item error:", error);
      return c.json({ error: "Internal server error" }, 500);
    }
  }
);

/* ------------------------- REMOVE ITEM FROM BASKET ------------------------- */
basketRouter.delete("/items/:itemId", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");
  const itemId = c.req.param("itemId");

  try {
    // Verify user owns this basket item
    const basketItem = await prisma.basketItem.findFirst({
      where: {
        id: itemId,
        basket: { userId: user.id },
      },
    });

    if (!basketItem) {
      return c.json({ error: "Basket item not found" }, 404);
    }

    await prisma.basketItem.delete({
      where: { id: itemId },
    });

    return c.json({ message: "Item removed from basket" });
  } catch (error) {
    console.error("Remove from basket error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

/* ------------------------- CLEAR BASKET ------------------------- */
basketRouter.delete("/", async (c) => {
  const prisma = c.get("prisma");
  const user = c.get("user");

  try {
    const basket = await prisma.basket.findFirst({
      where: { userId: user.id },
    });

    if (basket) {
      await prisma.basketItem.deleteMany({
        where: { basketId: basket.id },
      });
    }

    return c.json({ message: "Basket cleared successfully" });
  } catch (error) {
    console.error("Clear basket error:", error);
    return c.json({ error: "Internal server error" }, 500);
  }
});

export default basketRouter;
