import { config } from "../../config/config.js";
import type { AuthUser } from "@/types/auth.js";
import { compare, hash } from "bcryptjs";
import { createMiddleware } from "hono/factory";
import { sign, verify } from "hono/jwt";

const JWT_SECRET = config.jwt.secret;
const BCRYPT_ROUNDS = config.bcrypt.rounds;
const JWT_EXPIRES_IN = config.jwt.expiresIn;

// Password helpers
export const hashPassword = (password: string) => hash(password, BCRYPT_ROUNDS);
export const verifyPassword = (password: string, hashed: string) =>
  compare(password, hashed);

// Sign token using config
export const generateToken = (user: AuthUser): Promise<string> => {
  return sign(
    {
      id: user.id,
      email: user.email,
      role: user.role,
      exp: Math.floor(Date.now() / 1000) + JWT_EXPIRES_IN,
    },
    JWT_SECRET
  );
};

export const verifyToken = (token: string): AuthUser | null => {
  try {
    const payload = verify(token, JWT_SECRET);
    return payload as unknown as AuthUser;
  } catch {
    return null;
  }
};

// JWT middleware
export const jwtMiddleware = createMiddleware(async (c, next) => {
  const authHeader = c.req.header("Authorization");
  if (!authHeader?.startsWith("Bearer ")) {
    return await next();
  }

  const token = authHeader.slice(7);

  try {
    const payload = await verify(token, JWT_SECRET);
    c.set("jwtPayload", payload);
  } catch {}

  await next();
});

// Attach user to context
export const authMiddleware = createMiddleware(async (c, next) => {
  const payload = c.get("jwtPayload") as any;

  if (payload && payload.id && payload.email && payload.role) {
    c.set("user", {
      id: Number(payload.id),
      email: payload.email,
      role: payload.role,
    } as AuthUser);
  }

  await next();
});

// Require logged-in user
export const requireAuth = createMiddleware(async (c, next) => {
  const user = c.get("user") as AuthUser | undefined;
  if (!user) {
    return c.json({ error: "Unauthorized" }, 401);
  }
  await next();
});

// Require admin
export const requireAdmin = createMiddleware(async (c, next) => {
  const user = c.get("user") as AuthUser | undefined;
  if (!user || user.role !== "ADMIN") {
    return c.json({ error: "Admin access required" }, 403);
  }
  await next();
});
