export const openApiDoc = {
  openapi: "3.0.0",
  info: {
    title: "E-Commerce API",
    description:
      "Complete e-commerce platform with authentication, products, orders, and payments",
    version: "1.0.0",
    contact: {
      name: "API Support",
      email: "support@ecommerce.com",
    },
  },
  servers: [
    {
      url: "http://localhost:4000",
      description: "Development server",
    },
  ],
  components: {
    securitySchemes: {
      bearerAuth: {
        type: "http",
        scheme: "bearer",
        bearerFormat: "JWT",
      },
    },
    schemas: {
      // ... (all the schemas from previous response remain the same)
      Error: {
        type: "object",
        properties: {
          error: { type: "string" },
        },
      },
      Pagination: {
        type: "object",
        properties: {
          page: { type: "integer" },
          limit: { type: "integer" },
          total: { type: "integer" },
          pages: { type: "integer" },
          hasNext: { type: "boolean" },
          hasPrev: { type: "boolean" },
        },
      },
      HealthCheck: {
        type: "object",
        properties: {
          status: { type: "string", example: "OK" },
          timestamp: { type: "string", format: "date-time" },
        },
      },

      // Auth schemas
      RegisterRequest: {
        type: "object",
        required: ["email", "password", "name"],
        properties: {
          email: { type: "string", format: "email" },
          password: { type: "string", minLength: 6 },
          name: { type: "string" },
          phone: { type: "string" },
        },
      },
      LoginRequest: {
        type: "object",
        required: ["email", "password"],
        properties: {
          email: { type: "string", format: "email" },
          password: { type: "string" },
        },
      },
      AuthResponse: {
        type: "object",
        properties: {
          user: {
            type: "object",
            properties: {
              id: { type: "integer" },
              email: { type: "string" },
              name: { type: "string" },
              phone: { type: "string" },
              role: { type: "string", enum: ["USER", "ADMIN"] },
              createdAt: { type: "string", format: "date-time" },
              updatedAt: { type: "string", format: "date-time" },
            },
          },
          token: { type: "string" },
        },
      },
      ProfileUpdate: {
        type: "object",
        properties: {
          name: { type: "string" },
          phone: { type: "string" },
        },
      },
      ForgotPassword: {
        type: "object",
        required: ["email"],
        properties: {
          email: { type: "string", format: "email" },
        },
      },
      ResetPassword: {
        type: "object",
        required: ["token", "newPassword"],
        properties: {
          token: { type: "string" },
          newPassword: { type: "string", minLength: 6 },
        },
      },

      // Product schemas
      Product: {
        type: "object",
        properties: {
          id: { type: "integer" },
          name: { type: "string" },
          slug: { type: "string" },
          description: { type: "string" },
          price: { type: "number", format: "float" },
          discountPrice: { type: "number", format: "float" },
          status: {
            type: "string",
            enum: [
              "DRAFT",
              "ACTIVE",
              "INACTIVE",
              "OUT_OF_STOCK",
              "DISCONTINUED",
            ],
          },
          averageRating: { type: "number", format: "float" },
          reviewCount: { type: "integer" },
          category: { $ref: "#/components/schemas/Category" },
          images: {
            type: "array",
            items: { $ref: "#/components/schemas/ProductImage" },
          },
          variants: {
            type: "array",
            items: { $ref: "#/components/schemas/ProductVariant" },
          },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      ProductCreate: {
        type: "object",
        required: ["name", "slug", "price", "categoryId"],
        properties: {
          name: { type: "string" },
          slug: { type: "string" },
          description: { type: "string" },
          sku: { type: "string" },
          price: { type: "number", format: "float" },
          discountPrice: { type: "number", format: "float" },
          status: {
            type: "string",
            enum: [
              "DRAFT",
              "ACTIVE",
              "INACTIVE",
              "OUT_OF_STOCK",
              "DISCONTINUED",
            ],
            default: "DRAFT",
          },
          barcode: { type: "string" },
          categoryId: { type: "integer" },
          images: {
            type: "array",
            items: {
              type: "object",
              properties: {
                url: { type: "string", format: "uri" },
                altText: { type: "string" },
                isPrimary: { type: "boolean", default: false },
              },
            },
          },
          variants: {
            type: "array",
            items: {
              type: "object",
              required: ["name", "slug"],
              properties: {
                name: { type: "string" },
                slug: { type: "string" },
                sku: { type: "string" },
                price: { type: "number", format: "float" },
                stock: { type: "integer", default: 0 },
              },
            },
          },
          tagIds: {
            type: "array",
            items: { type: "integer" },
          },
        },
      },
      ProductImage: {
        type: "object",
        properties: {
          id: { type: "integer" },
          url: { type: "string", format: "uri" },
          altText: { type: "string" },
          isPrimary: { type: "boolean" },
        },
      },
      ProductVariant: {
        type: "object",
        properties: {
          id: { type: "integer" },
          name: { type: "string" },
          slug: { type: "string" },
          sku: { type: "string" },
          price: { type: "number", format: "float" },
          stock: { type: "integer" },
        },
      },

      // Category schemas
      Category: {
        type: "object",
        properties: {
          id: { type: "integer" },
          name: { type: "string" },
          slug: { type: "string" },
          description: { type: "string" },
          image: { type: "string" },
          parentId: { type: "integer" },
          parent: { $ref: "#/components/schemas/Category" },
          children: {
            type: "array",
            items: { $ref: "#/components/schemas/Category" },
          },
          _count: {
            type: "object",
            properties: {
              products: { type: "integer" },
            },
          },
        },
      },
      CategoryCreate: {
        type: "object",
        required: ["name", "slug"],
        properties: {
          name: { type: "string" },
          slug: { type: "string" },
          description: { type: "string" },
          image: { type: "string" },
          parentId: { type: "integer" },
        },
      },

      // Basket schemas
      Basket: {
        type: "object",
        properties: {
          id: { type: "integer" },
          userId: { type: "integer" },
          items: {
            type: "array",
            items: { $ref: "#/components/schemas/BasketItem" },
          },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      BasketItem: {
        type: "object",
        properties: {
          id: { type: "string" },
          productId: { type: "integer" },
          variantId: { type: "integer" },
          quantity: { type: "integer" },
          product: { $ref: "#/components/schemas/Product" },
          variant: { $ref: "#/components/schemas/ProductVariant" },
        },
      },
      AddToBasket: {
        type: "object",
        required: ["productId"],
        properties: {
          productId: { type: "integer" },
          variantId: { type: "integer" },
          quantity: { type: "integer", default: 1, minimum: 1 },
        },
      },
      UpdateBasketItem: {
        type: "object",
        required: ["quantity"],
        properties: {
          quantity: { type: "integer", minimum: 0 },
        },
      },

      // Address schemas
      Address: {
        type: "object",
        properties: {
          id: { type: "integer" },
          firstName: { type: "string" },
          lastName: { type: "string" },
          company: { type: "string" },
          addressLine1: { type: "string" },
          addressLine2: { type: "string" },
          city: { type: "string" },
          state: { type: "string" },
          postalCode: { type: "string" },
          country: { type: "string", default: "US" },
          phone: { type: "string" },
          isDefault: { type: "boolean", default: false },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      AddressCreate: {
        type: "object",
        required: [
          "firstName",
          "lastName",
          "addressLine1",
          "city",
          "state",
          "postalCode",
        ],
        properties: {
          firstName: { type: "string" },
          lastName: { type: "string" },
          company: { type: "string" },
          addressLine1: { type: "string" },
          addressLine2: { type: "string" },
          city: { type: "string" },
          state: { type: "string" },
          postalCode: { type: "string" },
          country: { type: "string", default: "US" },
          phone: { type: "string" },
          isDefault: { type: "boolean", default: false },
        },
      },

      // Order schemas
      Order: {
        type: "object",
        properties: {
          id: { type: "string" },
          orderNumber: { type: "string" },
          userId: { type: "integer" },
          status: {
            type: "string",
            enum: [
              "PENDING",
              "CONFIRMED",
              "PROCESSING",
              "SHIPPED",
              "DELIVERED",
              "CANCELLED",
              "REFUNDED",
            ],
          },
          totalAmount: { type: "number", format: "float" },
          finalAmount: { type: "number", format: "float" },
          items: {
            type: "array",
            items: { $ref: "#/components/schemas/OrderItem" },
          },
          shippingAddress: { $ref: "#/components/schemas/Address" },
          billingAddress: { $ref: "#/components/schemas/Address" },
          payments: {
            type: "array",
            items: { $ref: "#/components/schemas/Payment" },
          },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      OrderItem: {
        type: "object",
        properties: {
          id: { type: "string" },
          productId: { type: "integer" },
          variantId: { type: "integer" },
          quantity: { type: "integer" },
          unitPrice: { type: "number", format: "float" },
          totalPrice: { type: "number", format: "float" },
          product: { $ref: "#/components/schemas/Product" },
          variant: { $ref: "#/components/schemas/ProductVariant" },
        },
      },
      OrderCreate: {
        type: "object",
        required: ["shippingAddressId", "paymentMethod"],
        properties: {
          shippingAddressId: { type: "integer" },
          billingAddressId: { type: "integer" },
          paymentMethod: { type: "string" },
          notes: { type: "string" },
        },
      },
      OrderStatusUpdate: {
        type: "object",
        required: ["status"],
        properties: {
          status: {
            type: "string",
            enum: [
              "PENDING",
              "CONFIRMED",
              "PROCESSING",
              "SHIPPED",
              "DELIVERED",
              "CANCELLED",
              "REFUNDED",
            ],
          },
        },
      },

      // Payment schemas
      Payment: {
        type: "object",
        properties: {
          id: { type: "string" },
          orderId: { type: "string" },
          userId: { type: "integer" },
          paymentMethod: { type: "string" },
          amount: { type: "number", format: "float" },
          status: {
            type: "string",
            enum: ["PENDING", "COMPLETED", "FAILED", "REFUNDED"],
          },
          paymentIntentId: { type: "string" },
          transactionId: { type: "string" },
          currency: { type: "string", default: "USD" },
          paidAt: { type: "string", format: "date-time" },
          createdAt: { type: "string", format: "date-time" },
        },
      },
      PaymentIntentCreate: {
        type: "object",
        required: ["orderId", "paymentMethod"],
        properties: {
          orderId: { type: "string" },
          paymentMethod: { type: "string" },
        },
      },
      PaymentConfirm: {
        type: "object",
        required: ["paymentIntentId"],
        properties: {
          paymentIntentId: { type: "string" },
          transactionId: { type: "string" },
          metadata: { type: "object" },
        },
      },
      PaymentIntentResponse: {
        type: "object",
        properties: {
          paymentIntentId: { type: "string" },
          clientSecret: { type: "string" },
          amount: { type: "number", format: "float" },
          currency: { type: "string" },
        },
      },

      // Review schemas
      Review: {
        type: "object",
        properties: {
          id: { type: "integer" },
          userId: { type: "integer" },
          productId: { type: "integer" },
          rating: { type: "integer", minimum: 1, maximum: 5 },
          title: { type: "string" },
          comment: { type: "string" },
          isVerified: { type: "boolean" },
          user: {
            type: "object",
            properties: {
              name: { type: "string" },
            },
          },
          product: {
            type: "object",
            properties: {
              name: { type: "string" },
              slug: { type: "string" },
            },
          },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      ReviewCreate: {
        type: "object",
        required: ["productId", "rating"],
        properties: {
          productId: { type: "integer" },
          rating: { type: "integer", minimum: 1, maximum: 5 },
          title: { type: "string" },
          comment: { type: "string" },
        },
      },

      // Wishlist schemas
      Wishlist: {
        type: "object",
        properties: {
          id: { type: "integer" },
          userId: { type: "integer" },
          items: {
            type: "array",
            items: { $ref: "#/components/schemas/WishlistItem" },
          },
          createdAt: { type: "string", format: "date-time" },
          updatedAt: { type: "string", format: "date-time" },
        },
      },
      WishlistItem: {
        type: "object",
        properties: {
          id: { type: "string" },
          productId: { type: "integer" },
          product: { $ref: "#/components/schemas/Product" },
          createdAt: { type: "string", format: "date-time" },
        },
      },
      AddToWishlist: {
        type: "object",
        required: ["productId"],
        properties: {
          productId: { type: "integer" },
        },
      },
      // Adding missing schemas for completeness
      InventoryUpdate: {
        type: "object",
        required: ["quantity", "movementType"],
        properties: {
          variantId: { type: "integer" },
          quantity: { type: "integer" },
          movementType: {
            type: "string",
            enum: ["PURCHASE", "SALE", "RETURN", "ADJUSTMENT", "TRANSFER"],
          },
          reason: { type: "string" },
        },
      },
      InventoryMovement: {
        type: "object",
        properties: {
          id: { type: "integer" },
          productId: { type: "integer" },
          variantId: { type: "integer" },
          movementType: { type: "string" },
          quantity: { type: "integer" },
          previousStock: { type: "integer" },
          newStock: { type: "integer" },
          reason: { type: "string" },
          createdAt: { type: "string", format: "date-time" },
        },
      },
    },
  },
  paths: {
    // ... (all previous paths remain the same)
    "/health": {
      get: {
        summary: "Health Check",
        description: "Check if the API is running",
        tags: ["System"],
        responses: {
          200: {
            description: "API is healthy",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/HealthCheck" },
              },
            },
          },
        },
      },
    },

    // Public Routes
    "/public/health": {
      get: {
        summary: "Public Health Check",
        description: "Public health check endpoint",
        tags: ["Public"],
        responses: {
          200: {
            description: "Service is healthy",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/HealthCheck" },
              },
            },
          },
        },
      },
    },

    // Authentication Routes
    "/auth/register": {
      post: {
        summary: "Register User",
        description: "Create a new user account",
        tags: ["Authentication"],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/RegisterRequest" },
            },
          },
        },
        responses: {
          201: {
            description: "User created successfully",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/AuthResponse" },
              },
            },
          },
          400: {
            description: "Bad request",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
          500: {
            description: "Internal server error",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/auth/login": {
      post: {
        summary: "Login User",
        description: "Authenticate user and return JWT token",
        tags: ["Authentication"],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/LoginRequest" },
            },
          },
        },
        responses: {
          200: {
            description: "Login successful",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/AuthResponse" },
              },
            },
          },
          401: {
            description: "Invalid credentials",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/auth/profile": {
      get: {
        summary: "Get User Profile",
        description: "Get current user profile information",
        tags: ["Authentication"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Profile retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    user: {
                      type: "object",
                      properties: {
                        id: { type: "integer" },
                        email: { type: "string" },
                        name: { type: "string" },
                        phone: { type: "string" },
                        role: { type: "string" },
                        createdAt: { type: "string", format: "date-time" },
                        updatedAt: { type: "string", format: "date-time" },
                      },
                    },
                  },
                },
              },
            },
          },
        },
      },
      patch: {
        summary: "Update User Profile",
        description: "Update current user profile information",
        tags: ["Authentication"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ProfileUpdate" },
            },
          },
        },
        responses: {
          200: {
            description: "Profile updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    user: {
                      type: "object",
                      properties: {
                        id: { type: "integer" },
                        email: { type: "string" },
                        name: { type: "string" },
                        phone: { type: "string" },
                        createdAt: { type: "string", format: "date-time" },
                        updatedAt: { type: "string", format: "date-time" },
                      },
                    },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/auth/logout": {
      post: {
        summary: "Logout User",
        description: "Logout user and invalidate sessions",
        tags: ["Authentication"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Logout successful",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/auth/forgot-password": {
      post: {
        summary: "Forgot Password",
        description: "Request password reset token",
        tags: ["Authentication"],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ForgotPassword" },
            },
          },
        },
        responses: {
          200: {
            description: "Reset token sent if email exists",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/auth/reset-password": {
      post: {
        summary: "Reset Password",
        description: "Reset password using token",
        tags: ["Authentication"],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ResetPassword" },
            },
          },
        },
        responses: {
          200: {
            description: "Password reset successful",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          400: {
            description: "Invalid or expired token",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Public Products Routes
    "/public/products": {
      get: {
        summary: "List Products",
        description:
          "Get paginated list of products with filtering and sorting",
        tags: ["Public Products"],
        parameters: [
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1, minimum: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 20, minimum: 1, maximum: 100 },
          },
          {
            name: "category",
            in: "query",
            schema: { type: "string" },
          },
          {
            name: "search",
            in: "query",
            schema: { type: "string" },
          },
          {
            name: "minPrice",
            in: "query",
            schema: { type: "number", minimum: 0 },
          },
          {
            name: "maxPrice",
            in: "query",
            schema: { type: "number", minimum: 0 },
          },
          {
            name: "sortBy",
            in: "query",
            schema: {
              type: "string",
              enum: ["name", "price", "createdAt", "popularity"],
              default: "createdAt",
            },
          },
          {
            name: "sortOrder",
            in: "query",
            schema: {
              type: "string",
              enum: ["asc", "desc"],
              default: "desc",
            },
          },
        ],
        responses: {
          200: {
            description: "Products retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    products: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Product" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/public/products/slug/{slug}": {
      get: {
        summary: "Get Product by Slug",
        description: "Get product details by slug",
        tags: ["Public Products"],
        parameters: [
          {
            name: "slug",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Product retrieved successfully",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Product" },
              },
            },
          },
          404: {
            description: "Product not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/public/products/{id}": {
      get: {
        summary: "Get Product by ID",
        description: "Get product details by ID",
        tags: ["Public Products"],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Product retrieved successfully",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Product" },
              },
            },
          },
          404: {
            description: "Product not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Public Categories Routes
    "/public/categories": {
      get: {
        summary: "List Categories",
        description: "Get all top-level categories with subcategories",
        tags: ["Public Categories"],
        responses: {
          200: {
            description: "Categories retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    categories: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Category" },
                    },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/public/categories/slug/{slug}": {
      get: {
        summary: "Get Category by Slug",
        description: "Get category details by slug with products",
        tags: ["Public Categories"],
        parameters: [
          {
            name: "slug",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Category retrieved successfully",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Category" },
              },
            },
          },
        },
      },
    },

    // Protected Products Routes (Admin)
    "/products": {
      post: {
        summary: "Create Product",
        description: "Create a new product (Admin only)",
        tags: ["Products"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ProductCreate" },
            },
          },
        },
        responses: {
          201: {
            description: "Product created successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    product: { $ref: "#/components/schemas/Product" },
                  },
                },
              },
            },
          },
        },
      },
      get: {
        summary: "List Products (Admin)",
        description: "Get paginated list of products for admin",
        tags: ["Products"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 20 },
          },
          {
            name: "status",
            in: "query",
            schema: { type: "string" },
          },
          {
            name: "categoryId",
            in: "query",
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Products retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    products: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Product" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/products/{id}": {
      put: {
        summary: "Update Product",
        description: "Update product details (Admin only)",
        tags: ["Products"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ProductCreate" },
            },
          },
        },
        responses: {
          200: {
            description: "Product updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    product: { $ref: "#/components/schemas/Product" },
                  },
                },
              },
            },
          },
        },
      },
      delete: {
        summary: "Delete Product",
        description: "Delete a product (Admin only)",
        tags: ["Products"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Product deleted successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    // Basket Routes
    "/basket": {
      get: {
        summary: "Get Basket",
        description: "Get current user's shopping basket",
        tags: ["Basket"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Basket retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    basket: { $ref: "#/components/schemas/Basket" },
                  },
                },
              },
            },
          },
        },
      },
      delete: {
        summary: "Clear Basket",
        description: "Remove all items from basket",
        tags: ["Basket"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Basket cleared successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/basket/items": {
      post: {
        summary: "Add Item to Basket",
        description: "Add product to shopping basket",
        tags: ["Basket"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/AddToBasket" },
            },
          },
        },
        responses: {
          201: {
            description: "Item added to basket",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    item: { $ref: "#/components/schemas/BasketItem" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/basket/items/{itemId}": {
      put: {
        summary: "Update Basket Item",
        description: "Update basket item quantity",
        tags: ["Basket"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "itemId",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/UpdateBasketItem" },
            },
          },
        },
        responses: {
          200: {
            description: "Item updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    item: { $ref: "#/components/schemas/BasketItem" },
                  },
                },
              },
            },
          },
        },
      },
      delete: {
        summary: "Remove Item from Basket",
        description: "Remove item from shopping basket",
        tags: ["Basket"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "itemId",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Item removed from basket",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    // Orders Routes
    "/orders": {
      post: {
        summary: "Create Order",
        description: "Create a new order from basket",
        tags: ["Orders"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/OrderCreate" },
            },
          },
        },
        responses: {
          201: {
            description: "Order created successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    order: { $ref: "#/components/schemas/Order" },
                  },
                },
              },
            },
          },
        },
      },
      get: {
        summary: "Get User Orders",
        description: "Get paginated list of user orders",
        tags: ["Orders"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 10 },
          },
        ],
        responses: {
          200: {
            description: "Orders retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    orders: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Order" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
        },
      },
    },
    // Orders Routes (continued)
    "/orders/{id}": {
      get: {
        summary: "Get Order Details",
        description: "Get detailed information about a specific order",
        tags: ["Orders"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Order details retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    order: { $ref: "#/components/schemas/Order" },
                  },
                },
              },
            },
          },
          404: {
            description: "Order not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/orders/{id}/cancel": {
      post: {
        summary: "Cancel Order",
        description: "Cancel a pending or confirmed order",
        tags: ["Orders"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Order cancelled successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    order: { $ref: "#/components/schemas/Order" },
                  },
                },
              },
            },
          },
          400: {
            description: "Order cannot be cancelled",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Admin Orders Routes
    "/orders/admin/orders": {
      get: {
        summary: "Get All Orders (Admin)",
        description: "Get paginated list of all orders (Admin only)",
        tags: ["Orders", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 20 },
          },
          {
            name: "status",
            in: "query",
            schema: {
              type: "string",
              enum: [
                "PENDING",
                "CONFIRMED",
                "PROCESSING",
                "SHIPPED",
                "DELIVERED",
                "CANCELLED",
                "REFUNDED",
              ],
            },
          },
        ],
        responses: {
          200: {
            description: "Orders retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    orders: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Order" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/orders/admin/orders/{id}/status": {
      patch: {
        summary: "Update Order Status (Admin)",
        description: "Update order status (Admin only)",
        tags: ["Orders", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/OrderStatusUpdate" },
            },
          },
        },
        responses: {
          200: {
            description: "Order status updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    order: { $ref: "#/components/schemas/Order" },
                  },
                },
              },
            },
          },
        },
      },
    },

    // Payments Routes
    "/payments/intent": {
      post: {
        summary: "Create Payment Intent",
        description: "Create a payment intent for an order",
        tags: ["Payments"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/PaymentIntentCreate" },
            },
          },
        },
        responses: {
          200: {
            description: "Payment intent created successfully",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/PaymentIntentResponse" },
              },
            },
          },
          404: {
            description: "Order not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/payments/confirm": {
      post: {
        summary: "Confirm Payment",
        description: "Confirm and process a payment",
        tags: ["Payments"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/PaymentConfirm" },
            },
          },
        },
        responses: {
          200: {
            description: "Payment confirmed successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    payment: { $ref: "#/components/schemas/Payment" },
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          404: {
            description: "Payment not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/payments/{id}": {
      get: {
        summary: "Get Payment Status",
        description: "Get payment details and status",
        tags: ["Payments"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Payment details retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    payment: { $ref: "#/components/schemas/Payment" },
                  },
                },
              },
            },
          },
          404: {
            description: "Payment not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/payments/admin/{id}/refund": {
      post: {
        summary: "Refund Payment (Admin)",
        description: "Refund a completed payment (Admin only)",
        tags: ["Payments", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "string" },
          },
        ],
        responses: {
          200: {
            description: "Payment refunded successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    payment: { $ref: "#/components/schemas/Payment" },
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          400: {
            description: "Only completed payments can be refunded",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Addresses Routes
    "/addresses": {
      get: {
        summary: "Get User Addresses",
        description: "Get all addresses for the current user",
        tags: ["Addresses"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Addresses retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    addresses: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Address" },
                    },
                  },
                },
              },
            },
          },
        },
      },
      post: {
        summary: "Create Address",
        description: "Create a new address for the current user",
        tags: ["Addresses"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/AddressCreate" },
            },
          },
        },
        responses: {
          201: {
            description: "Address created successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    address: { $ref: "#/components/schemas/Address" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/addresses/{id}": {
      put: {
        summary: "Update Address",
        description: "Update an existing address",
        tags: ["Addresses"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/AddressCreate" },
            },
          },
        },
        responses: {
          200: {
            description: "Address updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    address: { $ref: "#/components/schemas/Address" },
                  },
                },
              },
            },
          },
          404: {
            description: "Address not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
      delete: {
        summary: "Delete Address",
        description: "Delete an address",
        tags: ["Addresses"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Address deleted successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          404: {
            description: "Address not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/addresses/{id}/default": {
      patch: {
        summary: "Set Default Address",
        description: "Set an address as the default shipping address",
        tags: ["Addresses"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Address set as default successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    address: { $ref: "#/components/schemas/Address" },
                  },
                },
              },
            },
          },
          404: {
            description: "Address not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Reviews Routes
    "/reviews": {
      post: {
        summary: "Create Review",
        description: "Create a review for a purchased product",
        tags: ["Reviews"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ReviewCreate" },
            },
          },
        },
        responses: {
          201: {
            description: "Review created successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    review: { $ref: "#/components/schemas/Review" },
                  },
                },
              },
            },
          },
          400: {
            description:
              "Cannot review unpurchased product or duplicate review",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/reviews/my-reviews": {
      get: {
        summary: "Get User Reviews",
        description: "Get paginated list of current user's reviews",
        tags: ["Reviews"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 10 },
          },
        ],
        responses: {
          200: {
            description: "Reviews retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    reviews: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Review" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/reviews/{id}": {
      put: {
        summary: "Update Review",
        description: "Update an existing review",
        tags: ["Reviews"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/ReviewCreate" },
            },
          },
        },
        responses: {
          200: {
            description: "Review updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    review: { $ref: "#/components/schemas/Review" },
                  },
                },
              },
            },
          },
          404: {
            description: "Review not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
      delete: {
        summary: "Delete Review",
        description: "Delete a review",
        tags: ["Reviews"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Review deleted successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          404: {
            description: "Review not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Wishlist Routes
    "/wishlist": {
      get: {
        summary: "Get Wishlist",
        description: "Get current user's wishlist",
        tags: ["Wishlist"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Wishlist retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    wishlist: { $ref: "#/components/schemas/Wishlist" },
                  },
                },
              },
            },
          },
        },
      },
      delete: {
        summary: "Clear Wishlist",
        description: "Remove all items from wishlist",
        tags: ["Wishlist"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Wishlist cleared successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/wishlist/items": {
      post: {
        summary: "Add to Wishlist",
        description: "Add a product to wishlist",
        tags: ["Wishlist"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/AddToWishlist" },
            },
          },
        },
        responses: {
          201: {
            description: "Product added to wishlist",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    item: { $ref: "#/components/schemas/WishlistItem" },
                  },
                },
              },
            },
          },
          400: {
            description: "Product already in wishlist",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    "/wishlist/items/{productId}": {
      delete: {
        summary: "Remove from Wishlist",
        description: "Remove a product from wishlist",
        tags: ["Wishlist"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "productId",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Product removed from wishlist",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          404: {
            description: "Wishlist or product not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Categories Routes (Admin)
    "/categories": {
      post: {
        summary: "Create Category (Admin)",
        description: "Create a new product category (Admin only)",
        tags: ["Categories", "Admin"],
        security: [{ bearerAuth: [] }],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/CategoryCreate" },
            },
          },
        },
        responses: {
          201: {
            description: "Category created successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    category: { $ref: "#/components/schemas/Category" },
                  },
                },
              },
            },
          },
          400: {
            description: "Category slug already exists",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
      get: {
        summary: "List Categories (Admin)",
        description: "Get all categories with details (Admin only)",
        tags: ["Categories", "Admin"],
        security: [{ bearerAuth: [] }],
        responses: {
          200: {
            description: "Categories retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    categories: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Category" },
                    },
                  },
                },
              },
            },
          },
        },
      },
    },

    "/categories/{id}": {
      put: {
        summary: "Update Category (Admin)",
        description: "Update category details (Admin only)",
        tags: ["Categories", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/CategoryCreate" },
            },
          },
        },
        responses: {
          200: {
            description: "Category updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    category: { $ref: "#/components/schemas/Category" },
                  },
                },
              },
            },
          },
        },
      },
      delete: {
        summary: "Delete Category (Admin)",
        description: "Delete a category (Admin only)",
        tags: ["Categories", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Category deleted successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                  },
                },
              },
            },
          },
          400: {
            description:
              "Cannot delete category with products or subcategories",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Products Inventory Routes
    "/products/{id}/inventory": {
      patch: {
        summary: "Update Inventory (Admin)",
        description: "Update product inventory stock (Admin only)",
        tags: ["Products", "Admin"],
        security: [{ bearerAuth: [] }],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        requestBody: {
          required: true,
          content: {
            "application/json": {
              schema: { $ref: "#/components/schemas/InventoryUpdate" },
            },
          },
        },
        responses: {
          200: {
            description: "Inventory updated successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    message: { type: "string" },
                    previousStock: { type: "integer" },
                    newStock: { type: "integer" },
                  },
                },
              },
            },
          },
          404: {
            description: "Product or variant not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Public Categories Products Route
    "/public/categories/{id}/products": {
      get: {
        summary: "Get Category Products",
        description:
          "Get paginated products for a category and its subcategories",
        tags: ["Public Categories"],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
          {
            name: "page",
            in: "query",
            schema: { type: "integer", default: 1 },
          },
          {
            name: "limit",
            in: "query",
            schema: { type: "integer", default: 20 },
          },
        ],
        responses: {
          200: {
            description: "Category products retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    category: {
                      type: "object",
                      properties: {
                        id: { type: "integer" },
                        name: { type: "string" },
                        slug: { type: "string" },
                      },
                    },
                    products: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Product" },
                    },
                    pagination: { $ref: "#/components/schemas/Pagination" },
                  },
                },
              },
            },
          },
          404: {
            description: "Category not found",
            content: {
              "application/json": {
                schema: { $ref: "#/components/schemas/Error" },
              },
            },
          },
        },
      },
    },

    // Public Products Related Route
    "/public/products/{id}/related": {
      get: {
        summary: "Get Related Products",
        description: "Get products related to the specified product",
        tags: ["Public Products"],
        parameters: [
          {
            name: "id",
            in: "path",
            required: true,
            schema: { type: "integer" },
          },
        ],
        responses: {
          200: {
            description: "Related products retrieved successfully",
            content: {
              "application/json": {
                schema: {
                  type: "object",
                  properties: {
                    products: {
                      type: "array",
                      items: { $ref: "#/components/schemas/Product" },
                    },
                  },
                },
              },
            },
          },
        },
      },
    },
  },
};
